// Language Courses Website - Version 4 (HTML/CSS/JS)

// Languages data
const languages = [
  { 
    id: 'english', 
    name: 'Английский', 
    icon: 'icons/course-globe.svg', 
    description: 'Самый популярный язык'
  },
  { 
    id: 'spanish', 
    name: 'Испанский', 
    icon: 'icons/course-languages.svg', 
    description: 'Язык страсти'
  },
  { 
    id: 'chinese', 
    name: 'Китайский', 
    icon: 'icons/course-book.svg', 
    description: 'Откройте Азию'
  },
  { 
    id: 'german', 
    name: 'Немецкий', 
    icon: 'icons/course-award.svg', 
    description: 'Язык бизнеса'
  }
]

// Initialize page
function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'courses'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initCoursesPage()
  }
  
  initSlider()
}

// Initialize courses page
function initCoursesPage() {
  loadCourses()
  updateStats()
  
  // Poll for storage changes
  setInterval(() => {
    loadCourses()
    updateStats()
  }, 500)
}

// Load courses from localStorage
function loadCourses() {
  const saved = localStorage.getItem('languageCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  } else {
    // Default courses
    courses = [
      {
        id: 1,
        language: 'Английский',
        level: 'Intermediate',
        progress: 65,
        rating: 5,
        icon: 'icons/course-globe.svg'
      },
      {
        id: 2,
        language: 'Испанский',
        level: 'Beginner',
        progress: 30,
        rating: 4,
        icon: 'icons/course-languages.svg'
      },
      {
        id: 3,
        language: 'Китайский',
        level: 'Advanced',
        progress: 85,
        rating: 5,
        icon: 'icons/course-book.svg'
      }
    ]
    localStorage.setItem('languageCourses', JSON.stringify(courses))
  }
  
  return courses
}

// Update stats
function updateStats() {
  const courses = loadCourses()
  const stats = [
    { 
      value: courses.length, 
      label: 'Курсов',
      icon: 'icons/stat-book.svg',
      color: '#A78BFA'
    },
    { 
      value: courses.length > 0 ? Math.round(courses.reduce((sum, c) => sum + c.progress, 0) / courses.length) : 0, 
      label: 'Прогресс',
      icon: 'icons/stat-progress.svg',
      color: '#A78BFA'
    },
    { 
      value: courses.length > 0 ? (courses.reduce((sum, c) => sum + c.rating, 0) / courses.length).toFixed(1) : 0, 
      label: 'Оценка',
      icon: 'icons/stat-rating.svg',
      color: '#A78BFA'
    }
  ]
  
  const statsMinimal = document.getElementById('statsMinimal')
  if (statsMinimal) {
    statsMinimal.innerHTML = stats.map(stat => `
      <div class="stat-minimal">
        <div class="stat-minimal-icon">
          <img src="${stat.icon}" alt="" width="24" height="24" />
        </div>
        <div class="stat-minimal-content">
          <div class="stat-minimal-value">${stat.value}${stat.label.includes('Прогресс') ? '%' : ''}</div>
          <div class="stat-minimal-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }

  renderCourses()
}

// Render courses
function renderCourses() {
  const courses = loadCourses()
  const coursesVertical = document.getElementById('coursesVertical')
  
  if (!coursesVertical) return
  
  if (courses.length === 0) {
    coursesVertical.innerHTML = `
      <div class="empty-state">
        <img src="icons/logo-book.svg" alt="" width="56" height="56" />
        <p>У вас пока нет активных курсов</p>
      </div>
    `
    return
  }
  
  coursesVertical.innerHTML = courses.map((course, index) => {
    const stars = Array.from({ length: 5 }, (_, i) => 
      i < course.rating ? '★' : '☆'
    ).join('')
    
    return `
      <div class="course-item">
        <div class="course-item-left">
          <div class="course-item-icon">
            <img src="${course.icon || 'icons/course-globe.svg'}" alt="${course.language}" width="28" height="28" />
          </div>
          <div class="course-item-info">
            <h3 class="course-item-language">${course.language}</h3>
            <p class="course-item-level">${course.level}</p>
          </div>
        </div>
        <div class="course-item-center">
          <div class="progress-minimal">
            <div class="progress-minimal-bar">
              <div class="progress-minimal-fill" style="width: ${course.progress}%"></div>
            </div>
            <span class="progress-minimal-text">${course.progress}%</span>
          </div>
        </div>
        <div class="course-item-right">
          <div class="stars-minimal">${stars}</div>
        </div>
      </div>
    `
  }).join('')
}

// Initialize enroll page
function initEnrollPage() {
  renderLanguages()
  
  const form = document.getElementById('enrollmentForm')
  const dateInput = document.getElementById('date')
  let selectedLanguage = ''
  
  // Set date limits
  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  dateInput.min = today.toISOString().split('T')[0]
  dateInput.max = maxDate.toISOString().split('T')[0]
  
  // Language selection
  document.querySelectorAll('.language-item').forEach(item => {
    item.addEventListener('click', function() {
      document.querySelectorAll('.language-item').forEach(i => i.classList.remove('selected'))
      this.classList.add('selected')
      selectedLanguage = this.dataset.language
    })
  })
  
  // Form submit
  form.addEventListener('submit', function(e) {
    e.preventDefault()
    
    if (!selectedLanguage || !dateInput.value) {
      alert('Пожалуйста, выберите язык и дату старта')
      return
    }
    
    const savedCourses = localStorage.getItem('languageCourses')
    const courses = savedCourses ? JSON.parse(savedCourses) : []
    
    const languageData = languages.find(l => l.id === selectedLanguage)
    const newCourse = {
      id: Date.now(),
      language: languageData.name,
      level: 'Beginner',
      progress: 0,
      rating: 0,
      icon: languageData.icon
    }

    courses.push(newCourse)
    localStorage.setItem('languageCourses', JSON.stringify(courses))
    
    // Show success message
    const successMsg = document.getElementById('successMessage')
    successMsg.style.display = 'flex'
    
    setTimeout(() => {
      window.location.href = 'index.html'
    }, 2000)
  })
}

// Render languages
function renderLanguages() {
  const languagesList = document.getElementById('languagesList')
  if (!languagesList) return
  
  languagesList.innerHTML = languages.map(lang => `
    <button type="button" class="language-item" data-language="${lang.id}">
      <div class="language-item-icon">
        <img src="${lang.icon}" alt="${lang.name}" width="24" height="24" />
      </div>
      <div class="language-item-content">
        <h3 class="language-item-name">${lang.name}</h3>
        <p class="language-item-desc">${lang.description}</p>
      </div>
    </button>
  `).join('')
}

// Initialize slider
function initSlider() {
  const slider = document.getElementById('slider')
  if (!slider) return
  
  const slides = slider.querySelectorAll('.slide')
  const dots = slider.querySelectorAll('.dot')
  const prevBtn = document.getElementById('sliderPrev')
  const nextBtn = document.getElementById('sliderNext')
  
  let currentIndex = 0
  
  function showSlide(index) {
    slides.forEach((slide, i) => {
      slide.classList.toggle('active', i === index)
    })
    dots.forEach((dot, i) => {
      dot.classList.toggle('active', i === index)
    })
  }
  
  function nextSlide() {
    currentIndex = (currentIndex + 1) % slides.length
    showSlide(currentIndex)
  }
  
  function prevSlide() {
    currentIndex = (currentIndex - 1 + slides.length) % slides.length
    showSlide(currentIndex)
  }
  
  if (prevBtn) prevBtn.addEventListener('click', prevSlide)
  if (nextBtn) nextBtn.addEventListener('click', nextSlide)
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => {
      currentIndex = index
      showSlide(currentIndex)
    })
  })
  
  // Auto-advance every 3 seconds
  setInterval(nextSlide, 3000)
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

